/*
 * Decompiled with CFR 0.152.
 */
package nl.altindag.ssl.util;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.security.Key;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.function.UnaryOperator;
import javax.net.ssl.X509TrustManager;
import nl.altindag.ssl.exception.GenericKeyStoreException;
import nl.altindag.ssl.util.CertificateUtils;
import nl.altindag.ssl.util.StringUtils;
import nl.altindag.ssl.util.ValidationUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class KeyStoreUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(KeyStoreUtils.class);
    public static final String DUMMY_PASSWORD = "dummy-password";
    private static final String KEYSTORE_TYPE = "PKCS12";
    private static final String EMPTY_INPUT_STREAM_EXCEPTION_MESSAGE = "Failed to load the keystore from the provided InputStream because it is null";
    private static final UnaryOperator<String> KEYSTORE_NOT_FOUND_EXCEPTION_MESSAGE = certificatePath -> String.format("Failed to load the keystore from the classpath for the given path: [%s]", certificatePath);
    private static final String EMPTY_TRUST_MANAGER_FOR_TRUSTSTORE_EXCEPTION = "Could not create TrustStore because the provided TrustManager does not contain any trusted certificates";
    private static final String EMPTY_CERTIFICATES_EXCEPTION = "Could not create TrustStore because certificate is absent";

    private KeyStoreUtils() {
    }

    public static KeyStore loadKeyStore(String keystorePath, char[] keystorePassword) {
        return KeyStoreUtils.loadKeyStore(keystorePath, keystorePassword, KeyStore.getDefaultType());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore loadKeyStore(String keystorePath, char[] keystorePassword, String keystoreType) {
        try (InputStream keystoreInputStream = KeyStoreUtils.class.getClassLoader().getResourceAsStream(keystorePath);){
            KeyStore keyStore = KeyStoreUtils.loadKeyStore(ValidationUtils.requireNotNull(keystoreInputStream, (String)KEYSTORE_NOT_FOUND_EXCEPTION_MESSAGE.apply(keystorePath)), keystorePassword, keystoreType);
            return keyStore;
        }
        catch (Exception e) {
            throw new GenericKeyStoreException(e);
        }
    }

    public static KeyStore loadKeyStore(Path keystorePath, char[] keystorePassword) {
        return KeyStoreUtils.loadKeyStore(keystorePath, keystorePassword, KeyStore.getDefaultType());
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static KeyStore loadKeyStore(Path keystorePath, char[] keystorePassword, String keystoreType) {
        try (InputStream keystoreInputStream = Files.newInputStream(keystorePath, StandardOpenOption.READ);){
            KeyStore keyStore = KeyStoreUtils.loadKeyStore(keystoreInputStream, keystorePassword, keystoreType);
            return keyStore;
        }
        catch (Exception e) {
            throw new GenericKeyStoreException(e);
        }
    }

    public static KeyStore loadKeyStore(InputStream keystoreInputStream, char[] keystorePassword) {
        return KeyStoreUtils.loadKeyStore(ValidationUtils.requireNotNull(keystoreInputStream, EMPTY_INPUT_STREAM_EXCEPTION_MESSAGE), keystorePassword, KeyStore.getDefaultType());
    }

    public static KeyStore loadKeyStore(InputStream keystoreInputStream, char[] keystorePassword, String keystoreType) {
        try {
            KeyStore keystore = KeyStore.getInstance(keystoreType);
            keystore.load(ValidationUtils.requireNotNull(keystoreInputStream, EMPTY_INPUT_STREAM_EXCEPTION_MESSAGE), keystorePassword);
            return keystore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new GenericKeyStoreException(e);
        }
    }

    public static KeyStore createIdentityStore(Key privateKey, char[] privateKeyPassword, String alias, List<? extends Certificate> certificateChain) {
        return KeyStoreUtils.createIdentityStore((Key)privateKey, (char[])privateKeyPassword, (String)alias, (Certificate[])certificateChain.toArray(new Certificate[0]));
    }

    public static KeyStore createIdentityStore(Key privateKey, char[] privateKeyPassword, List<? extends Certificate> certificateChain) {
        return KeyStoreUtils.createIdentityStore((Key)privateKey, (char[])privateKeyPassword, null, (Certificate[])certificateChain.toArray(new Certificate[0]));
    }

    @SafeVarargs
    public static <T extends Certificate> KeyStore createIdentityStore(Key privateKey, char[] privateKeyPassword, T ... certificateChain) {
        return KeyStoreUtils.createIdentityStore((Key)privateKey, (char[])privateKeyPassword, null, certificateChain);
    }

    @SafeVarargs
    public static <T extends Certificate> KeyStore createIdentityStore(Key privateKey, char[] privateKeyPassword, String alias, T ... certificateChain) {
        try {
            KeyStore keyStore = KeyStoreUtils.createKeyStore();
            String privateKeyAlias = StringUtils.isBlank(alias) ? CertificateUtils.generateAlias(certificateChain[0]) : alias;
            keyStore.setKeyEntry(privateKeyAlias, privateKey, privateKeyPassword, (Certificate[])certificateChain);
            return keyStore;
        }
        catch (KeyStoreException e) {
            throw new GenericKeyStoreException(e);
        }
    }

    public static KeyStore createKeyStore() {
        return KeyStoreUtils.createKeyStore(DUMMY_PASSWORD.toCharArray());
    }

    public static KeyStore createKeyStore(char[] keyStorePassword) {
        return KeyStoreUtils.createKeyStore(KEYSTORE_TYPE, keyStorePassword);
    }

    public static KeyStore createKeyStore(String keyStoreType, char[] keyStorePassword) {
        try {
            KeyStore keyStore = KeyStore.getInstance(keyStoreType);
            keyStore.load(null, keyStorePassword);
            return keyStore;
        }
        catch (IOException | KeyStoreException | NoSuchAlgorithmException | CertificateException e) {
            throw new GenericKeyStoreException(e);
        }
    }

    @SafeVarargs
    public static <T extends X509TrustManager> KeyStore createTrustStore(T ... trustManagers) {
        ArrayList<X509Certificate> certificates = new ArrayList<X509Certificate>();
        for (T trustManager : trustManagers) {
            certificates.addAll(Arrays.asList(trustManager.getAcceptedIssuers()));
        }
        return KeyStoreUtils.createTrustStore(ValidationUtils.requireNotEmpty(certificates, EMPTY_TRUST_MANAGER_FOR_TRUSTSTORE_EXCEPTION));
    }

    @SafeVarargs
    public static <T extends Certificate> KeyStore createTrustStore(T ... certificates) {
        return KeyStoreUtils.createTrustStore(Arrays.asList(certificates));
    }

    public static <T extends Certificate> KeyStore createTrustStore(List<T> certificates) {
        try {
            KeyStore trustStore = KeyStoreUtils.createKeyStore();
            for (Certificate certificate : ValidationUtils.requireNotEmpty(certificates, EMPTY_CERTIFICATES_EXCEPTION)) {
                String alias = CertificateUtils.generateAlias(certificate);
                boolean shouldAddCertificate = true;
                if (trustStore.containsAlias(alias)) {
                    for (int number = 0; number <= 1000; ++number) {
                        String mayBeUniqueAlias = alias + "-" + number;
                        if (!trustStore.containsAlias(mayBeUniqueAlias)) {
                            alias = mayBeUniqueAlias;
                            shouldAddCertificate = true;
                            break;
                        }
                        shouldAddCertificate = false;
                    }
                }
                if (!shouldAddCertificate) continue;
                trustStore.setCertificateEntry(alias, certificate);
            }
            return trustStore;
        }
        catch (KeyStoreException e) {
            throw new GenericKeyStoreException(e);
        }
    }

    public static List<KeyStore> loadSystemKeyStores() {
        ArrayList<KeyStore> keyStores = new ArrayList<KeyStore>();
        String operatingSystem = System.getProperty("os.name").toLowerCase();
        if (operatingSystem.contains("windows")) {
            KeyStore windowsRootKeyStore = KeyStoreUtils.createKeyStore("Windows-ROOT", null);
            KeyStore windowsMyKeyStore = KeyStoreUtils.createKeyStore("Windows-MY", null);
            keyStores.add(windowsRootKeyStore);
            keyStores.add(windowsMyKeyStore);
        }
        if (operatingSystem.contains("mac")) {
            KeyStore macKeyStore = KeyStoreUtils.createKeyStore("KeychainStore", null);
            keyStores.add(macKeyStore);
        }
        if (operatingSystem.contains("linux")) {
            String javaVendor = System.getProperty("java.vendor", "").toLowerCase();
            String javaVmVendor = System.getProperty("java.vm.vendor", "").toLowerCase();
            String javaRuntimeName = System.getProperty("java.runtime.name", "").toLowerCase();
            if (javaVendor.equals("the android project") || javaVmVendor.equals("the android project") || javaRuntimeName.equals("android runtime")) {
                KeyStore androidCAStore = KeyStoreUtils.createKeyStore("AndroidCAStore", null);
                keyStores.add(androidCAStore);
            }
        }
        if (keyStores.isEmpty()) {
            LOGGER.warn("No system KeyStores available for [{}]", (Object)operatingSystem);
            return Collections.emptyList();
        }
        return Collections.unmodifiableList(keyStores);
    }

    public static int countAmountOfTrustMaterial(KeyStore keyStore) {
        return KeyStoreUtils.amountOfSpecifiedMaterial(keyStore, KeyStore::isCertificateEntry, Integer.MAX_VALUE);
    }

    public static int countAmountOfIdentityMaterial(KeyStore keyStore) {
        return KeyStoreUtils.amountOfSpecifiedMaterial(keyStore, KeyStore::isKeyEntry, Integer.MAX_VALUE);
    }

    public static boolean containsTrustMaterial(KeyStore keyStore) {
        return KeyStoreUtils.amountOfSpecifiedMaterial(keyStore, KeyStore::isCertificateEntry, 1) > 0;
    }

    public static boolean containsIdentityMaterial(KeyStore keyStore) {
        return KeyStoreUtils.amountOfSpecifiedMaterial(keyStore, KeyStore::isKeyEntry, 1) > 0;
    }

    private static int amountOfSpecifiedMaterial(KeyStore keyStore, KeyStoreBiPredicate<KeyStore, String> predicate, int upperBoundaryForMaterialCounter) {
        try {
            int materialCounter = 0;
            Enumeration<String> aliases = keyStore.aliases();
            while (aliases.hasMoreElements() && materialCounter < upperBoundaryForMaterialCounter) {
                String alias = aliases.nextElement();
                if (!predicate.test(keyStore, alias)) continue;
                ++materialCounter;
            }
            return materialCounter;
        }
        catch (KeyStoreException e) {
            throw new GenericKeyStoreException(e);
        }
    }

    private static interface KeyStoreBiPredicate<T extends KeyStore, U> {
        public boolean test(T var1, U var2) throws KeyStoreException;
    }
}

